import { Router } from 'express';
import { z } from 'zod';
import { pool } from './db';

const router = Router();

// Accept the frontend payload but store as unit/category_id
const productSchema = z.object({
  name: z.string().min(1),
  quantityType: z.enum(['piece', 'dozen']),
  price: z.number().positive(),
  category: z.string().optional().nullable(),
  description: z.string().optional().nullable(),
});

function rowToProduct(row: any) {
  // Map DB unit pc/dz -> API quantityType piece/dozen, and include category name
  const quantityType = row.unit === 'dz' ? 'dozen' : 'piece';
  return {
    id: String(row.id),
    name: row.name,
    quantityType,
    price: Number(row.price),
    category: row.category_name ?? undefined,
    description: row.description ?? undefined,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

router.get('/', async (_req, res) => {
  const [rows] = await pool.query(`
    SELECT p.*, c.name AS category_name
    FROM products p
    LEFT JOIN categories c ON c.id = p.category_id
    ORDER BY p.id DESC
  `);
  const list = Array.isArray(rows) ? rows.map(rowToProduct) : [];
  res.json(list);
});

router.get('/:id', async (req, res) => {
  const [rows] = await pool.query(
    `SELECT p.*, c.name AS category_name FROM products p LEFT JOIN categories c ON c.id=p.category_id WHERE p.id = ?`,
    [req.params.id]
  );
  const list = Array.isArray(rows) ? rows.map(rowToProduct) : [];
  if (list.length === 0) return res.status(404).json({ error: 'Not found' });
  res.json(list[0]);
});

router.post('/', async (req, res) => {
  const parse = productSchema.safeParse(req.body);
  if (!parse.success) return res.status(400).json({ error: parse.error.flatten() });
  const data = parse.data;
  const unit = data.quantityType === 'dozen' ? 'dz' : 'pc';

  // Resolve category name -> id if provided
  let category_id: number | null = null;
  if (data.category && data.category.trim()) {
    const [catRows] = await pool.query('SELECT id FROM categories WHERE name = ?', [data.category.trim()]);
    if (Array.isArray(catRows) && catRows.length > 0) {
      // @ts-ignore
      category_id = Number(catRows[0].id);
    } else {
      const [ins] = await pool.query('INSERT INTO categories (name) VALUES (?)', [data.category.trim()]);
      // @ts-ignore
      category_id = Number(ins.insertId);
    }
  }

  const [result] = await pool.query(
    'INSERT INTO products (name, unit, price, category_id, description) VALUES (?, ?, ?, ?, ?)',
    [data.name, unit, data.price, category_id, data.description ?? null]
  );
  // @ts-ignore - OkPacket type differs
  const id = String(result.insertId);
  const [rows] = await pool.query(
    'SELECT p.*, c.name AS category_name FROM products p LEFT JOIN categories c ON c.id=p.category_id WHERE p.id = ?',
    [id]
  );
  const list = Array.isArray(rows) ? rows.map(rowToProduct) : [];
  res.status(201).json(list[0]);
});

router.put('/:id', async (req, res) => {
  const parse = productSchema.safeParse(req.body);
  if (!parse.success) return res.status(400).json({ error: parse.error.flatten() });
  const data = parse.data;
  const unit = data.quantityType === 'dozen' ? 'dz' : 'pc';

  let category_id: number | null = null;
  if (data.category && data.category.trim()) {
    const [catRows] = await pool.query('SELECT id FROM categories WHERE name = ?', [data.category.trim()]);
    if (Array.isArray(catRows) && catRows.length > 0) {
      // @ts-ignore
      category_id = Number(catRows[0].id);
    } else {
      const [ins] = await pool.query('INSERT INTO categories (name) VALUES (?)', [data.category.trim()]);
      // @ts-ignore
      category_id = Number(ins.insertId);
    }
  }

  const [result] = await pool.query(
    'UPDATE products SET name=?, unit=?, price=?, category_id=?, description=? WHERE id=?',
    [data.name, unit, data.price, category_id, data.description ?? null, req.params.id]
  );
  // @ts-ignore - OkPacket
  if (result.affectedRows === 0) return res.status(404).json({ error: 'Not found' });
  const [rows] = await pool.query(
    'SELECT p.*, c.name AS category_name FROM products p LEFT JOIN categories c ON c.id=p.category_id WHERE p.id = ?',
    [req.params.id]
  );
  const list = Array.isArray(rows) ? rows.map(rowToProduct) : [];
  res.json(list[0]);
});

router.delete('/:id', async (req, res) => {
  const [result] = await pool.query('DELETE FROM products WHERE id = ?', [req.params.id]);
  // @ts-ignore - OkPacket
  if (result.affectedRows === 0) return res.status(404).json({ error: 'Not found' });
  res.status(204).end();
});

export default router;
