import { Router } from 'express';
import { z } from 'zod';
import { pool } from './db';

const router = Router();

const categorySchema = z.object({
  name: z.string().min(1),
  description: z.string().optional().nullable(),
});

function rowToCategory(row: any) {
  return {
    id: String(row.id),
    name: row.name,
    description: row.description ?? undefined,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

router.get('/', async (_req, res) => {
  const [rows] = await pool.query('SELECT * FROM categories ORDER BY name ASC');
  const list = Array.isArray(rows) ? rows.map(rowToCategory) : [];
  res.json(list);
});

router.post('/', async (req, res) => {
  const parse = categorySchema.safeParse(req.body);
  if (!parse.success) return res.status(400).json({ error: parse.error.flatten() });
  const data = parse.data;
  const [result] = await pool.query(
    'INSERT INTO categories (name, description) VALUES (?, ?)',
    [data.name, data.description ?? null]
  );
  // @ts-ignore
  const id = String(result.insertId);
  const [rows] = await pool.query('SELECT * FROM categories WHERE id = ?', [id]);
  const list = Array.isArray(rows) ? rows.map(rowToCategory) : [];
  res.status(201).json(list[0]);
});

router.put('/:id', async (req, res) => {
  const parse = categorySchema.safeParse(req.body);
  if (!parse.success) return res.status(400).json({ error: parse.error.flatten() });
  const data = parse.data;
  const [result] = await pool.query(
    'UPDATE categories SET name=?, description=? WHERE id=?',
    [data.name, data.description ?? null, req.params.id]
  );
  // @ts-ignore
  if (result.affectedRows === 0) return res.status(404).json({ error: 'Not found' });
  const [rows] = await pool.query('SELECT * FROM categories WHERE id = ?', [req.params.id]);
  const list = Array.isArray(rows) ? rows.map(rowToCategory) : [];
  res.json(list[0]);
});

router.delete('/:id', async (req, res) => {
  const [result] = await pool.query('DELETE FROM categories WHERE id = ?', [req.params.id]);
  // @ts-ignore
  if (result.affectedRows === 0) return res.status(404).json({ error: 'Not found' });
  res.status(204).end();
});

export default router;